/******************************************************************************
*                       BC727X演示程序STM32版(无操作系统)                     *
*                                                                             *
* 程序说明：配合BC727x演示板，本程序为最简演示程序, 演示基本显示和读取键盘    *
            操作，适用于BC7276, BC7277芯片。                                  *
* CPU: STM32F103                                                              *
* RTOS: 无                                                                    *
* Main Frequency: 72MHz                                                       *
* 版权所有: 北京凌志比高科技有限公司 Bitcode Technology                       *
* 版本号: V2.0                                                                *
* 初始日期：2014年7月26日                                                     *
* 版本历史:                                                                   *
*   2014年7月   V1.0                                                          *
*   2017年3月   V2.0  重新改写程序使更容易阅读，使用EXTI和定时器中断处理按键  *
*                     键盘处理部分增加了长按键处理能力                        *
******************************************************************************/
#include "stm32f10x.h"

/* BC727X 指令（寄存器地址）*/
#define SEG_BLINK_CTL       0x10    // 断闪烁控制寄存器起始地址
#define DIG_BLINK_CTL_L     0x18    // 0-7位闪烁控制寄存器起始地址
#define DIG_BLINK_CTL_H     0x19    // 8-15位闪烁控制寄存器起始地址
#define BLINK_SPEED         0x1A    // 闪烁速度控制寄存器地址
#define DECODE_HEX          0x1b    // 16进制显示译码寄存器地址
#define DECODE_SEG          0x1c    // 段寻址译码寄存器地址
#define WRITE_ALL           0x1d    // 全局操作寄存器地址
#define DUMMY_CMD           0xff    // 假指令

/*   BC727X驱动使用SPI1口      */
/*   BC727X IS AT SPI1         */
/*   PA4 --- CS                */
/*   PA5 --- SCK               */
/*   PA6 --- MISO              */
/*   PA7 --- MOSI              */
/*   PC4 --- KEY               */

// I/O引脚定义 I/O pin definition
#define BC727X_CS       0x00000010      // CS at PA4
#define BC727X_KEY      0x00000010      // KEY at PC4

uint32_t KeyStatusChanged, LongPressKey;

// 初始化RCC函数 function to initialize RCC
void Init_RCC(void)
{
    // 调整APB2时钟频率以适合BC727X所要求速率
    // adjust APB2 clock rate to adapt BC727X SPI frequency
    RCC->CFGR |= RCC_CFGR_PPRE2_DIV8;       // APB2 clock = HCLK/8 (9MHz)
    
    // 使能 AFIO, IOPA, IOPC, SPI1
    RCC->APB2ENR |= RCC_APB2ENR_AFIOEN | RCC_APB2ENR_IOPAEN | RCC_APB2ENR_IOPCEN | RCC_APB2ENR_SPI1EN;
    
    // 使能TIM2 enable TIM2
    RCC->APB1ENR |= RCC_APB1ENR_TIM2EN;
}


// 初始化GPIO函数 function to initialize GPIOs
void Init_GPIO(void)
{
    // CRL/CRH most used values:
    // 4 -- float input (default)
    // 3 -- push-pull 50MHz
    // 2 -- push-pull 2MHz
    // 1 -- push-pull 10MHz
    // B -- alternate push-pull 50MHz
    // A -- alternate push-pull 2MHz
    // 9 -- alternate push-pull 10MHz
    // 8 -- input with pull-up/pull-down
    // 0 -- analog input
    
    GPIOA->CRL = 0xA8A24444;    // CLK, MOSI设为外设输出，CS为GPIO，MISO为上拉/下拉输入
                                // CLK, MOSI set as alternate push-pull output, CS as GPIO, MISO as pull-up/pull-down input
    GPIOA->ODR = 0x00000050;    // 设MISO(PA6)为上拉，CS(PA4)设为高电平
                                // MOSI(PA6) set as pull-up, CS(PA4) set to high
}

// 初始化SPI接口函数 function to initialize SPI1
void Init_SPI1(void)
{
    // SPI1配置为16位，pclk/256速率, 主机模式，clk空闲高电平，数据在第二个时钟沿采样
    // set SPI1 as 16-bit, frequency=pclk/256, master mode, clk idle high, data captured at 2nd clock phase
    // 最终SPI1时钟频率为35.156KHz
    // the actual SPI1 frequency is 35.156KHz
    SPI1->CR1 = SPI_CR1_DFF | SPI_CR1_SSM | SPI_CR1_SSI | SPI_CR1_SPE | SPI_CR1_BR 
                | SPI_CR1_MSTR | SPI_CR1_CPOL | SPI_CR1_CPHA;
}

// 初始化EXIT函数 function to initialize EXTI
void Init_EXTI(void)
{
    AFIO->EXTICR[1] = AFIO_EXTICR2_EXTI4_PC;    // 4号中断线使用 PC 口 EXTI line4 use Port C
    EXTI->RTSR = EXTI_RTSR_TR4;     // line4上升沿中断使能 rising edge on line4
    EXTI->FTSR = EXTI_FTSR_TR4;     // line4下降沿中断使能 falling edge on line4 
    EXTI->IMR = EXTI_IMR_MR4;       // 使能line4中断 enable line4 interrupt
}

// 初始化TIMER函数 function to initialize TIM2
void Init_TIM2(void)
{
    TIM2->PSC = 36000-1;  // 计数频率1000Hz set the counter frequency to 1000Hz
    TIM2->ARR = 2000;       // 计数溢出时间2s set the overflow time to 2s
    TIM2->DIER |= TIM_DIER_UIE; // 使能定时器溢出中断 enable timer interrupt
    TIM2->CR1 |= TIM_CR1_OPM | TIM_CR1_URS;   // TIM2设置为单次运行方式, 只有溢出产生中断
                                                // set to one pulse mode and only overflow generate an interrupt
    TIM2->EGR |= TIM_EGR_UG;    // 手动产生一个更新，从而使预分频器和自动重装寄存器得到更新
                                // generate an update event so the prescaler and auto-reload register can be updated
    TIM2->SR = 0;           // 清除中断标志 clear pending bit
}

// 初始化NVIC函数 function to initialize NVIC
void Init_NVIC(void)
{
    NVIC_EnableIRQ(EXTI4_IRQn);     // 使能EXTI4中断 enable EXTI4 interrupt
    NVIC_EnableIRQ(TIM2_IRQn);      // 使能TIM2中断 enable TIM2 interrupt
}


uint16_t spi_send(uint16_t data)
{
    GPIOA->BRR = BC727X_CS;        // CS = 0
    SPI1->DR = data;
    while ((SPI1->SR & SPI_SR_RXNE) == 0);		// 等待发送完成 wait for transmit to finish
    GPIOA->BSRR = BC727X_CS;       // CS = 1;
    return SPI1->DR;
}

void StartupDelay(void)
{
    volatile uint32_t i;
    for (i=0; i<0x8000; i++);
}

int main(void)
{
uint16_t keyMap;
uint16_t counter = 0;
    Init_RCC();
    Init_GPIO();
    Init_SPI1();
    Init_EXTI();
    Init_TIM2();
    Init_NVIC();
    KeyStatusChanged = 0;
    LongPressKey = 0;
    StartupDelay();         // 启动延时确保BC727X完成复位 delay on start to make sure BC727X complete reset process
    spi_send((uint16_t)WRITE_ALL<<8 | 0xFF);    // 清除显示 clear display
    while (1)
    {
        if (KeyStatusChanged)
        {
            KeyStatusChanged = 0;
            keyMap = spi_send(0xffff);
            if (keyMap != 0xffff)       // 如果有键按下 if any key is pressed
            {
                TIM2->EGR |= TIM_EGR_UG;    // 产生一更新事件，使计数器和预分频器清零 generate an update even, reset counter and prescaler
                TIM2->CR1 |= TIM_CR1_CEN;   // 启动定时器 start timer to detect long press
            }
            else
            {
                TIM2->CR1 &= ~TIM_CR1_CEN;  // 停止定时器 stop timer
            }
            spi_send(((uint16_t)DECODE_SEG<<8) | 0xA7);     // 清除长按键指示 clear indicator for long press
            spi_send(((uint16_t)DECODE_SEG<<8) | 0xAF);
            spi_send(((uint16_t)DECODE_SEG<<8) | 0xB7);
            spi_send(((uint16_t)DECODE_SEG<<8) | 0xBF);
            spi_send((((uint16_t)DECODE_HEX)<<8) | (0X70|((keyMap&0xf000)>>12)));    // 在第4-7位显示键盘映射值
            spi_send((((uint16_t)DECODE_HEX)<<8) | (0X60|((keyMap&0x0f00)>>8)));     // display key map at digit 4-7
            spi_send((((uint16_t)DECODE_HEX)<<8) | (0X50|((keyMap&0x00f0)>>4)));
            spi_send((((uint16_t)DECODE_HEX)<<8) | (0X40|(keyMap&0x000f)));
        }
        if (LongPressKey)   // 如果有长按键 if long press happened
        {
            LongPressKey = 0;
            spi_send(((uint16_t)DECODE_SEG<<8) | 0x27);     // 点亮第4-7位的小数点
            spi_send(((uint16_t)DECODE_SEG<<8) | 0x2F);     // turn on the decimal points of digit 4-7
            spi_send(((uint16_t)DECODE_SEG<<8) | 0x37);
            spi_send(((uint16_t)DECODE_SEG<<8) | 0x3F);
        }
        spi_send((((uint16_t)DECODE_HEX)<<8) | (0X00|(counter&0x000f)));    // 在第0-3位显示计数值
        if ((counter&0x000f) == 0)      // 只在第1位有更新时才更新该显示位，加快执行速度 only update display when contents changed
        {
            spi_send((((uint16_t)DECODE_HEX)<<8) | (0X10|((counter&0x00f0)>>4)));   // 更新第1位显示 update digit 1
            if ((counter&0x00ff) == 0)
            {
                spi_send((((uint16_t)DECODE_HEX)<<8) | (0X20|((counter&0x0f00)>>8)));   // 更新第2位显示 update digit 2
                if ((counter&0x0fff) == 0)
                {
                    spi_send((((uint16_t)DECODE_HEX)<<8) | (0X30|((counter&0xf000)>>12)));    // 更新第3位显示 update digit 3
                }
            }
        }
        counter++;
    }
}

// EXTI4中断服务程序 ISR for EXTI4
/*
外部中断4连接KEY引脚，当键盘状态变化时KEY引脚电平也发生变化
中断服务程序只设置键盘变化标志
EXTI line4 is connected with BC727X's KEY pin, the level on KEY changes as the keypad status changed.
the ISR only set the flag for keypad changing
*/
void EXTI4_IRQHandler(void)
{
    EXTI->PR |= EXTI_PR_PR4;    // 清除中断标志 clear line4 pending bit
    KeyStatusChanged = 1;      // 设置键盘变化标志 set the keypad changing flag
}

// TIM2中断服务程序 ISR for TIM2
/*
定时器6在有按键发生时由主程序启动，为单次运行方式
定时时间到后自动停止运行
TIM2 is started when there is key press in the main programme
It runs in one-pulse mode and stops at overflow
*/
void TIM2_IRQHandler(void)
{
    TIM2->SR = 0;   // 清除中断标志
    LongPressKey = 1;
}
