/******************************************************************************
*                       BC7591点阵显示演示程序 AVR 版
*
* 程序用途：BC7591 驱动点阵中英文显示演示
* CPU: AVR ATtiny416
* RTOS: 无
* Main Frequency: 内部20MHz(6分频)
* 版权所有: 北京凌志比高科技有限公司 https://bitcode.com.cn
* 版本号: V1.0
* 初始版本：2020年12月2日
* 版本历史:
*   2020年12月   V1.0
* 功能说明： 在32*8点阵上向左移动显示字符串"2020年9月15日"
*
* 使用环境： 本程序在Atmel Studio中使用AVR GCC调试通过。
* 程序说明：
*     本程序仅演示了显示功能，为使程序尽可能简化，未包括键盘功能，串口发送
*     部分也未使用中断。
*     配合BC7591驱动32*8LED点阵使用，点阵的接法同数据手册中典型电路
*     使用了精简小字库仅包括0-9数字及汉字“年月日”，并演示其用法
******************************************************************************/

#include <avr/io.h>
#include <stddef.h>

/*****************/
/*  PB2 --- TX   */
/*  PB3 --- RX   */
/*****************/

/**** BC759X 指令 ****/
// 寄存器操作指令
#define DIRECT_WRITE 0x00        // 直接写入显示寄存器，低5位为地址
#define SEG_OFF      0xC0        // 段熄灭
#define SEG_ON       0xC1        // 段点亮
#define WRITE_ALL    0xF1        // 全局写入，数据写入所有显示寄存器
// 数码管相关指令
#define DECODE_WT  0x80        // 译码显示，低5位为地址
#define WRITE_EXT  0xA8        // 扩展位不译码显示
#define DECODE_EXT 0xB0        // 扩展显示位译码显示，低2位为地址
// 点阵显示相关指令
#define COL_WRITE   0x00        // 列写入，低5位为地址
#define SHIFT_H_WT  0x40        // 插入并向右(高地址)平移，低5位为地址
#define SHIFT_L_WT  0x60        // 插入并向左(低地址)平移，低5位为地址
#define ROTATE_R    0x5F        // 向右(高地址)循环滚屏
#define ROTATE_L    0x60        // 向左(低地址)循环滚屏
#define QTR_WT_BOT  0xA0        // 写入底部 1/4 行，低2位为地址
#define QTR_INS_BOT 0xA4        // 底部1/4行插入写入，低2位为地址
#define QTR_WT_TOP  0xA8        // 写入顶部 1/4 行，低2位为地址
#define QTR_INS_TOP 0xAC        // 顶部1/4行插入写入，低2位为地址
#define COORD_OFF   0xC0        // 坐标点熄灭
#define COORD_ON    0xC1        // 坐标点点亮
// 控制指令
#define BLINK_WT_SET  0x30          // 段闪烁控制置1，低4位为地址
#define BLINK_WT_CLR  0x20          // 段闪烁控制置0，低4位为地址
#define BLINK_DIG_CTL 0xD0          // 位闪烁控制，
#define BLINK_SPEED   0xF2          // 闪烁速度控制
#define DIM_CTL       0xF3          // 亮度控制
#define GLOBAL_CTL    0xF0          // 整体控制(闪烁开/关，显示开/关)
#define RESET         0xFF5A        // 复位
#define UART_SEND_0   0xFFFF        // UART 发送 0x00

// 字体数据，可变字体宽度，第一个字节为字体宽度，后面是字体数据
const unsigned char En_space[] = { 0x03, 0x00, 0x00, 0x00 };                          // ' '
const unsigned char Ch_nian[]  = { 0x06, 0x24, 0xdc, 0x54, 0x7f, 0x54, 0x44 };        // '年'
const unsigned char Ch_yue[]   = { 0x06, 0x02, 0xfc, 0xa8, 0xa8, 0xa8, 0xfe };        // '月'
const unsigned char Ch_ri[]    = { 0x05, 0xfe, 0x92, 0x92, 0x92, 0xfe };              // '日'
const unsigned char En_0[]     = { 0x05, 0x7c, 0x8a, 0x92, 0xa2, 0x7c };              // '0'
const unsigned char En_1[]     = { 0x03, 0x42, 0xfe, 0x02 };                          // '1'
const unsigned char En_2[]     = { 0x05, 0x4e, 0x92, 0x92, 0x92, 0x62 };              // '2'
const unsigned char En_3[]     = { 0x05, 0x44, 0x92, 0x92, 0x92, 0x6c };              // '3'
const unsigned char En_4[]     = { 0x05, 0x18, 0x28, 0x48, 0xfe, 0x08 };              // '4'
const unsigned char En_5[]     = { 0x05, 0xf2, 0x92, 0x92, 0x92, 0x8c };              // '5'
const unsigned char En_6[]     = { 0x05, 0x7c, 0x92, 0x92, 0x92, 0x0c };              // '6'
const unsigned char En_7[]     = { 0x05, 0x80, 0x8e, 0x90, 0xa0, 0xc0 };              // '7'
const unsigned char En_8[]     = { 0x05, 0x6c, 0x92, 0x92, 0x92, 0x6c };              // '8'
const unsigned char En_9[]     = { 0x05, 0x60, 0x92, 0x92, 0x92, 0x7c };              // '9'

// 字库索引和字体数据入口地址，索引排列顺序不一定和实际字库中数据先后顺序一致，但索引和入口地址表二者数量应相等，、
// 数据一一对应，如索引第2个数据为'月'，对应入口地址表中第2个数据必须是'月'的字体数据入口地址Ch_yue
const wchar_t        Index[14]    = { L' ', L'年', L'月', L'日',        // 字库索引
    L'1', L'2', L'3', L'4', L'5',
    L'6', L'7', L'8', L'9', L'0' };
const unsigned char* Entrence[14] = { En_space, Ch_nian, Ch_yue, Ch_ri,        // 字体数据入口地址
    En_1, En_2, En_3, En_4, En_5,
    En_6, En_7, En_8, En_9, En_0 };

void io_init(void);
void uart_init(void);
void delay(void);
void bc759x_cmd_send(unsigned char Cmd, unsigned char Data);        // 发送一个完整指令
int  index_lookup(wchar_t Wchar);                                   // 查找字符在字库中的位置，返回值为位置，返回 -1 表示该字符不在字库中
void insert_str_L(unsigned char Pos, const wchar_t* Str);           // 在Pos列向左平移插入显示字符串Str

int main(void)
{
    unsigned char i;
    io_init();
    uart_init();
    for (i = 0; i < 5; i++)        // 上电延时，确保BC759X进入工作状态
    {
        delay();
    }
    bc759x_cmd_send(WRITE_ALL, 0xff);        // 亮点全部显示段
    for (i = 0; i < 20; i++)                 // 保持全亮一段时间
    {
        delay();
    }
    bc759x_cmd_send(WRITE_ALL, 0x00);        // 清除显示
    while (1)
    {
        insert_str_L(31, L"2020年9月15日");        // 显示字符串
        for (i = 0; i < 20; i++)                   // 停顿一段时间
        {
            delay();
        }
        bc759x_cmd_send(WRITE_ALL, 0x00);        // 清除显示
    }
}

void io_init(void)        // 设置 UART 所用I/O口
{
    PORTB.OUTSET   = 0x04;        // 设置 TX(PB2) 为高电平
    PORTB.DIRSET   = 0x04;        // 设置 TX(PB2) 为输出
    PORTB.DIRCLR   = 0x08;        // 设置 RX(PB3) 为输入
    PORTB.PIN3CTRL = 0x08;        // 设置 RX(PB3) 上拉
}

void uart_init(void)        // 设置 UART 工作模式和波特率
{
    USART0.BAUD  = 0x56D;        // ATTINY416默认工作主频3.333MHz，设置串口在普通工作模式下波特率为9600
    USART0.CTRLC = 0x03;         // 设置为 异步模式、无奇偶校验、1个停止位、8位数据位
    USART0.CTRLB = 0x40;         // 使能发送
    //    USART0.CTRLB = 0xC0;         // 使能接收和发送
    //    USART0.CTRLA = 0x80;         // 使能 RXCIE(接收)中断
}

void delay(void)
{
    volatile unsigned int i16;
    for (i16 = 0; i16 < 13000; i16++)
    {
    }
}

// 发送一个完整指令
void bc759x_cmd_send(unsigned char Cmd, unsigned char Data)
{
    while (!(USART0.STATUS & 0x20))
        ;        // 检查DREIF确定发送缓冲区是否可接收新数据
    USART0.TXDATAL = Cmd;
    while (!(USART0.STATUS & 0x20))
        ;        // 检查DREIF确定发送缓冲区是否可接收新数据
    USART0.TXDATAL = Data;
}

//字库索引查找
int index_lookup(wchar_t Wchar)
{
    int i, IndexCnt;
    IndexCnt = sizeof(Index) / sizeof(wchar_t);        // 计算索引大小
    for (i = 0; i < IndexCnt; i++)
    {
        if (Index[i] == Wchar)        // 如果找到所查找字符，则跳出
        {
            break;
        }
    }
    if (i != IndexCnt)        // 如果是中途跳出
    {
        return i;        // 返回索引位置
    }
    else
    {
        return -1;        // 如果未找到，则返回-1
    }
}

// 向左平移插入字符串
void insert_str_L(unsigned char Pos, const wchar_t* Str)
{
    unsigned int  p;
    unsigned int  Idx;
    unsigned char i;
    p = 0;
    while (*(Str + p) != L'\0')        // 输出直至字符串结尾
    {
        Idx = index_lookup(*(Str + p));
        if (Idx == -1)        // 如果字符不在字库中
        {
            Idx = index_lookup(L' ');        // 该字符用空格代替
        }
        bc759x_cmd_send(SHIFT_L_WT | Pos, 0);        // 字符前插入一个空列以和前面字符分开
        delay();                                     // 延迟一段时间，控制平移速度
        for (i = 0; i < *Entrence[Idx]; i++)         // 依次输出字体数据
        {
            bc759x_cmd_send(SHIFT_L_WT | Pos, *(Entrence[Idx] + i + 1));        // 字体数据写入显示屏
            delay();                                                            // 延迟一段时间，控制平移速度
        }
        p++;        // 移动到下一字符
    }
}
