/******************************************************************************
*              BC759x演示程序 AVR 版 (使用键盘驱动库)
*
* 程序用途：BC759x LED显示及键盘接口芯片演示，可直接用于BC7591开发板，如果
*           用于BC7595芯片，因该芯片显示只有6位，部分显示超出了显示范围，
*           需要修改显示的位置
* CPU: AVR ATtiny416
* RTOS: 无
* Main Frequency: 内部20MHz(6分频)
* 版权所有: 北京凌志比高科技有限公司 https://bitcode.com.cn
* 版本号: V3.0
* 初始版本：2020年9月25日
* 版本历史:
*   2020年9月   V1.0
*	2020年10月  V2.0
*   2021年3月   V3.0 改为使用键盘驱动库
* 功能说明： 提供3位十进制数字显示和1个16段光柱显示，使用3个按键，分别为
*		    "+" ， "-" ， "SHIFT"。数字为闪烁显示
*           1. 单独按 +/- 键，数字加一或减一
*           2. 长按 +/- 键3秒，停止闪烁，数字快速增加减小，释放按键后恢复闪烁
*           3. 同时按SHIFT和+/-键，调整光柱
*           4. 如果30秒无按键，显示亮度调暗，进入节能状态，按任意键后恢复
*
* 使用环境： 本程序在Atmel Studio中使用AVR GCC调试通过。
* 程序说明：
*	  程序使用定时器和串口两个中断。
*     驱动库位于 /key_scan/子目录中。关于驱动库的使用更多信息请查询 《UART单线键盘
*	  接口驱动库技术手册》。
*	  本示例设置驱动库工作于检测按键释放模式，是为了长按快进模式时能检测按键的释放。但
*	  程序并未对不同按键的释放做分别处理。
******************************************************************************/

#include "key_scan.h"
#include <avr/interrupt.h>
#include <avr/io.h>

/*********************************/
/*  PB2 (MCU TX) --> BC759x RX   */
/*  PB3 (MCU RX) --> BC759x TX   */
/*********************************/

/**** BC759X 指令 ****/
// 寄存器操作指令
#define DIRECT_WRITE 0x00        // 直接写入显示寄存器，低5位为地址
#define SEG_OFF      0xC0        // 段熄灭
#define SEG_ON       0xC1        // 段点亮
#define WRITE_ALL    0xF1        // 全局写入，数据写入所有显示寄存器
// 数码管相关指令
#define DECODE_WT  0x80        // 译码显示，低5位为地址
#define DECODE_EXT 0xB0        // 扩展显示位译码显示，低2位为地址
// 点阵显示相关指令
#define COL_WRITE  0x00        // 列写入，低5位为地址
#define SHIFT_H_WT 0x40        // 插入并向右(高地址)平移，低5位为地址
#define SHIFT_L_WT 0x60        // 插入并向左(低地址)平移，低5位为地址
#define ROTATE_R   0x5F        // 向右(高地址)循环滚屏
#define ROTATE_L   0x60        // 向左(低地址)循环滚屏
#define QTR_WT_BOT 0xA0        // 写入底部 1/4 行，低2为为地址
#define QTR_WT_TOP 0xA8        // 写入顶部 1/4 行，低2位为地址
#define COORD_OFF  0xC0        // 坐标点熄灭
#define COORD_ON   0xC1        // 坐标点点亮
// 控制指令
#define BLINK_WT_SET  0x30          // 段闪烁控制置1，低4位为地址
#define BLINK_WT_CLR  0x20          // 段闪烁控制置0，低4位为地址
#define BLINK_DIG_CTL 0xD0          // 位闪烁控制，
#define BLINK_SPEED   0xF2          // 闪烁速度控制
#define DIM_CTL       0xF3          // 亮度控制
#define GLOBAL_CTL    0xF0          // 整体控制(闪烁开/关，显示开/关)
#define RESET         0xFF5A        // 复位
#define UART_SEND_0   0xFFFF        // UART 发送 0x00

// 程序有关的常量定义
#define KEY_PLUS    5         // 定义5号键为“+”键
#define KEY_MINUS   6         // 定义6号键为“-”键
#define KEY_SHIFT   2         // 定义2号键为"SHIFT"键(组合键)
#define SHIFT_PLUS  96        // BC759X最大可能产生的键值为95, 将键值96-126用作表示组合键和长按键
#define SHIFT_MINUS 97        // 两个组合键分别分配键号96和97
#define LP_PLUS     98        // “+“键长按定义键值为98
#define LP_MINUS    99        // "-"键长按定义键值为99

void io_init(void);
void uart_init(void);
void timer_init(void);
void delay_64ms(unsigned char T);
void bc759x_cmd_send(unsigned char Cmd, unsigned char Data);        // 发送一个完整指令
void display_dec(unsigned char Pos, unsigned char Number);          // 在Pos显示位以10进制显示数字
void display_bar16(unsigned char Pos, unsigned int BarVal);         // 在Pos位显示16段光柱

/********** 定义组合键 ******************************************/
const unsigned char  shift_plus[]  = { 2, SHIFT_PLUS, KEY_SHIFT, KEY_PLUS };          // 组合键1,由SHIFT键和PLUS键组成，组合键键值定义为SHIFT_PLUS
const unsigned char  shift_minus[] = { 2, SHIFT_MINUS, KEY_SHIFT, KEY_MINUS };        // 组合键2,由SHIFT键和MINUS键组成，组合键键值定义为SHIFT_MINUS
const unsigned char* CBKeyList[]   = { shift_plus, shift_minus };
unsigned char        CBMap[2];        // 供键盘驱动库使用的组合键映射数组

/********** 定义长按键 *****************************************/
const unsigned char  lp_plus[]   = { KEY_PLUS, LP_PLUS };          // 定义"+"的长按键
const unsigned char  lp_minus[]  = { KEY_MINUS, LP_MINUS };        // 定义”-“的长按键
const unsigned char  nokey[]     = { 0xff, 0xff };                 // 定义”无按键“
const unsigned char* LPKeyList[] = { lp_plus, lp_minus, nokey };

volatile unsigned int DownCounter;           //	延时计时器
unsigned char         Value = 100;           //	显示的数值
unsigned int          Bar1  = 0x0001;        //	显示的光柱值

int main(void)
{
    unsigned char NoKeyTimer = 0;
    unsigned char NewKey;

    set_detect_mode(1);                           // 检测按键释放
    set_longpress_count(47);                    // 设置长按键时间为3秒
    def_combined_key(CBKeyList, CBMap, 2);        // 设置组合键
    def_longpress_key(LPKeyList, 3);              // 设置长按键和无按键检测
    io_init();
    uart_init();
    timer_init();
    sei();                                          // 使能全局中断
    delay_64ms(5);                                  // 延时320ms，确保BC759X进入工作状态
    bc759x_cmd_send(WRITE_ALL, 0xff);               // 亮点全部显示段
    delay_64ms(30);                                 // 显示 30*64ms=1920ms
    bc759x_cmd_send(WRITE_ALL, 0x00);               // 清除显示
    bc759x_cmd_send(BLINK_SPEED, 20);               // 设置闪烁速度
    bc759x_cmd_send(BLINK_WT_SET | 5, 0xff);        // 设置第8，9，10显示位为闪烁显示
    bc759x_cmd_send(BLINK_WT_SET | 6, 0xff);
    bc759x_cmd_send(BLINK_WT_SET | 7, 0xff);
    display_dec(5, Value);         // 在第5,6,7位上显示数值
    display_bar16(0, Bar1);        // 显示光柱
    while (1)
    {
        /**************************/
        // 其它任务处理可以放在这里
        /***************************/

        // *********** 按键处理 ***********
        if (is_key_changed())        // 如果有新按键变化
        {
            NewKey = get_key_value();
            switch (NewKey)
            {
            case KEY_PLUS:        // 如果是“+”键按下
                if (Value < 255)
                {
                    display_dec(5, ++Value);
                }
                break;
            case LP_PLUS:                                 // "+"键长按
                bc759x_cmd_send(GLOBAL_CTL, 0x01);        // 停止闪烁显示
                while (get_key_value() == LP_PLUS)        // 如果是长按键一直持续，Value快速增加
                {
                    if (Value < 255)
                    {
                        display_dec(5, ++Value);        // 数值加1,然后刷新数值显示
                    }
                    delay_64ms(2);        // 延时128ms
                }
                bc759x_cmd_send(GLOBAL_CTL, 0);        // 按键释放，恢复闪烁显示
                break;
            case KEY_MINUS:        // “-”键按下
                if (Value > 0)
                {
                    display_dec(5, --Value);
                }
                break;
            case LP_MINUS:                                 // "-"键长按
                bc759x_cmd_send(GLOBAL_CTL, 0x01);         // 停止闪烁显示
                while (get_key_value() == LP_MINUS)        // 如果是长按键一直持续，Value快速减小
                {
                    if (Value > 0)
                    {
                        display_dec(5, --Value);        // 刷新数值显示，然后数值减1
                    }
                    delay_64ms(2);        // 延时128ms
                }
                bc759x_cmd_send(GLOBAL_CTL, 0);        // 恢复闪烁显示
                break;
            case SHIFT_PLUS:        // 如果是SHIFT-PLUG组合键按下 (光柱1增加)
                Bar1 = (Bar1 << 1) | 0x01;
                display_bar16(0, Bar1);
                break;
            case SHIFT_MINUS:        // 如果是SHIFT-MINUS组合键按下 (光柱1减小)
                Bar1 = Bar1 >> 1;
                display_bar16(0, Bar1);
                break;
            case 0xFF:        // "无按键"的长按键，收到此键值表示距最后一次键盘事件超过了一个长按键时间没有任何键盘操作
                NoKeyTimer++;
                if (NoKeyTimer >= 10)        // 如果超过10倍长按键时间
                {
                    NoKeyTimer = 0;
                    bc759x_cmd_send(DIM_CTL, 0x0C);        // 显示亮度调为0x0C(进入节能状态)
                }
                break;
            default:
                break;
            }
            if (NewKey != 0xFF)        // 对除了"无按键"以外的任何键盘事件
            {
                NoKeyTimer = 0;
                bc759x_cmd_send(DIM_CTL, 0);        // 亮度调到最高(从节能状态恢复)
            }
        }
        // ********** 按键处理部分结束 ***********
    }
}

void io_init(void)        // 设置 UART 所用I/O口
{
    PORTB.OUTSET   = 0x04;        // 设置 TX(PB2) 为高电平
    PORTB.DIRSET   = 0x04;        // 设置 TX(PB2) 为输出
    PORTB.DIRCLR   = 0x08;        // 设置 RX(PB3) 为输入
    PORTB.PIN3CTRL = 0x08;        // 设置 RX(PB3) 上拉
}

void uart_init(void)        // 设置 UART 工作模式和波特率
{
    USART0.BAUD  = 0x56D;        // ATTINY416默认工作主频3.333MHz，设置串口在普通工作模式下波特率为9600
    USART0.CTRLC = 0x03;         // 设置为 异步模式、无奇偶校验、1个停止位、8位数据位
    USART0.CTRLB = 0xC0;         // 使能接收和发送
    USART0.CTRLA = 0x80;         // 使能 RXCIE(接收)中断
}

void timer_init(void)        // 设置 RTC 定时器
{
    RTC.CLKSEL     = 0x01;        // 选择 1KHz 内部时钟为信号源
    RTC.PITINTCTRL = 0x01;        // 使能周期中断
    while (RTC.PITSTATUS)
        ;                       // 确定 PITCTRLA 寄存器不忙
    RTC.PITCTRLA = 0x29;        // 使能周期中断功能，定为64时钟周期 (用作主程序中延时计时)
    while (RTC.STATUS & 0x01)
        ;                    // 确定 CTRLA 寄存器不忙
    RTC.CTRLA = 0x01;        // 设置分频率为 1, 使能 RTC
}

void delay_64ms(unsigned char T)
{
    DownCounter = T;
    while (0 != DownCounter)
        ;
}

// 发送一个完整指令
void bc759x_cmd_send(unsigned char Cmd, unsigned char Data)
{
    while (!(USART0.STATUS & 0x20))
        ;        // 检查DREIF确定发送缓冲区是否可接收新数据
    USART0.TXDATAL = Cmd;
    while (!(USART0.STATUS & 0x20))
        ;        // 检查DREIF确定发送缓冲区是否可接收新数据
    USART0.TXDATAL = Data;
}

// 在Pos显示位以10进制显示数字
void display_dec(unsigned char Pos, unsigned char Number)
{
    unsigned char x;
    unsigned char NotFirstDigit = 0;
    Pos &= 0x1f;             // 确保pos值在有效范围内
    x = Number / 100;        // 得到百位
    if (0 != x)
    {
        bc759x_cmd_send(DECODE_WT | Pos, x);        // 显示百位
        NotFirstDigit = 1;
    }
    else
    {
        bc759x_cmd_send(DIRECT_WRITE | Pos, 0);        // 首位如果为0，不显示
    }
    Number = Number % 100;
    x      = Number / 10;                 // 得到十位
    if (NotFirstDigit || (0 != x))        // 如果十位不为0或不是首位
    {
        bc759x_cmd_send(DECODE_WT | (Pos + 1), x);
    }
    else
    {
        bc759x_cmd_send(DIRECT_WRITE | (Pos + 1), 0);        // 如果为0且是首位，不显示
    }
    Number = Number % 10;        // 得到个位
    bc759x_cmd_send(DECODE_WT | (Pos + 2), Number);
}

// 在Pos位显示16段光柱
void display_bar16(unsigned char Pos, unsigned int Bar)
{
    Pos &= 0x1f;                                           // 确保Pos值在有效范围内
    bc759x_cmd_send(DIRECT_WRITE | Pos, Bar >> 8);         // 写光柱高8段
    bc759x_cmd_send(DIRECT_WRITE | (Pos + 1), Bar);        // 写光柱低8段
}

// RTC 周期中断处理 (每64ms发生一次)
ISR(RTC_PIT_vect)
{
    RTC.PITINTFLAGS = 0x01;        // 清除中断标志
    if (0 != DownCounter)          // 如果 DownCounter 不为0，每64ms减一(通用计时用)
    {
        --DownCounter;
    }
    long_press_tick();        // 呼叫键盘驱动库长按键计数函数
}

// UART 接收中断处理
ISR(USART0_RXC_vect)
{
    update_key_status(USART0.RXDATAL);        // 呼叫键盘驱动库键盘状态更新函数
}
