/******************************************************************************
*                       BC6xxxx演示程序 AVR 版
*
* 程序说明：BC6xxx键盘接口芯片演示，程序不依赖任何外设库等其它外部软件
*         但可和外设库一起使用。
* CPU: AVR ATtiny416
* RTOS: 无
* Main Frequency: 内部20MHz
* 版权所有: 北京凌志比高科技有限公司 https://bitcode.com.cn
* 版本号: V2.0
* 初始版本：2020年7月18日
* 版本历史:
*   2020年7月   V1.0
*   2020年10月  V2.0
* 功能说明： 使用BC6xxx芯片的键盘接口程序，程序定义3个键，"+", "-"和"SHIFT"，
*			演示6种键盘事件的处理：
*           1. 单键按下
*           2. 单键释放
*           3. 单键长按
*           4. 组合键
*           5. 组合键长按
*			6. 长时间无按键
* 使用环境： 本程序在Atmel Studio中使用AVR GCC调试通过。
* 程序说明：
*     程序使用定时器和串口接收两个中断，用于接收按键数据，完成长按键计时。
*     中断中仅进行按键状态标记，按键的处理由主程序完成。
*     BC6xxx使用了键码0-87. 程序中将2个组合键赋予键码125-126，这样主
*     程序中可以像处理普通按键一样处理组合键，无需额外程序。
*     定时器按设计的长按键时长产生中断，每接收到按键，定时器都重新开始计时，
*     当定时中断发生，则说明在这段时间内没有新的按键状态的变化，如果当前是
*     有按键按下的状态，就说明该按键是持续处在按下的状态。定时中断会设置长
*     按键标志，并重新发出按键信号。因此主程序中处理按键时只需再判断长按键
*     标志，即可区分是单次按键还是长按键。
*     长按键不仅可用于普通按键，还可用于组合键，同时也适用于长时间无按键状
*     态的检测。
******************************************************************************/

#include <avr/interrupt.h>
#include <avr/io.h>

void io_init(void);
void uart_init(void);
void timer_init(void);
void delay_64ms(unsigned char T);

volatile unsigned char Counter_3s;           // 长按键计时
volatile unsigned char IsLongPress;          // 按键长按标志
volatile unsigned int  DownCounter;          // 延时定时器
volatile unsigned char IdleCounter;          // 节能计时器(一段时间无操作后降低显示亮度)
volatile unsigned char LastKeyNumber;        // 最后一次键值
volatile unsigned char NewKey;               // 新按键键值

#define LONG_PRESS_TIME  3000        // 定义长按键时间为 3000ms
#define KEY_PLUS         5           // 定义5号键为“+”键
#define KEY_MINUS        6           // 定义6号键为“-”键
#define KEY_SHIFT        2           // 定义2号键位"SHIFT"键(组合键)
#define COMBINATION_KEY1 125         // BC6xxx最大可能产生的键值为87, 可将键值88-126用作表示组合键(127不可用)
#define COMBINATION_KEY2 126

// 定义组合键数据结构, 此处定义为2个键，如有需要可增加
typedef struct
{
    unsigned char key1 : 1;
    unsigned char key2 : 1;
} CompositKey;

CompositKey Ck1, Ck2;        // 定义2个组合键 ("SHIFT_+", "SHIFT_-")

#define CK1_KEY1 KEY_SHIFT        // 组合键1 第1个键为SHIFT键
#define CK1_KEY2 KEY_PLUS         // 组合键1 第2个键为"+"键
#define CK2_KEY1 KEY_SHIFT        // 组合键2 第1个键同样为SHIFT键
#define CK2_KEY2 KEY_MINUS        // 组合键2 第2个键为"-"键

int main(void)
{
    io_init();
    uart_init();
    timer_init();
    sei();                // 使能全局中断
    delay_64ms(5);        // 延时320ms，确保BC6xxx进入工作状态
    while (1)
    {
        /**************************/
        // 其它任务处理可以放在这里
        /***************************/

        // *********** 按键处理 ***********
        if (NewKey != 0xFF)        // 如果有新按键变化
        {
            switch (NewKey)
            {
            case KEY_PLUS:        // 如果是“+”键按下 (单键按下处理)
                // 单键按下处理放在这里
                printf("Key Code %d , PLUS key ON\n", NewKey);
                if (IsLongPress == 1)        // 如果是单键长按键
                {
                    while (IsLongPress == 1)        // 检查长按键是否依然有效
                    {
                        // 单键长按键处理放在这里
                        printf("PLUS KEY long pressed\n");
                        delay_1ms(100);
                    }
                }
                break;
            case (KEY_PLUS | 0X80):          // "+"键释放
                if (IsLongPress == 0)        // 如果不是来自长按键定时自动重发
                {
                    printf("Key	Code %d	, PLUS key OFF\n", NewKey);
                }
                break;
            case KEY_MINUS:
                printf("Key Code %d ,	MINUS key ON\n", NewKey);
                break;
            case (KEY_MINUS | 0x80):         // 如果是“-”键释放 (单键释放处理)
                if (IsLongPress == 1)        // 如果要处理特定按键释放后达到一定时间的操作，放在这里
                {
                    // '-'键释放后经过超过3s, 期间没有其它按键操作
                    while (IsLongPress == 1)
                    {
                        // 如果要在"-"键释放后没有其它按键的状态3s后持续某种操作，放在这里
                        printf("MINUS key has been released for	more than 3s, no other keyboard	operations.\n");
                        delay_1ms(100);
                    }
                }
                else
                {
                    // 单键释放处理放在这里, 按键刚释放时的操作
                    printf("Key Code %d , MINUS key	OFF\n", NewKey);
                }
                break;
            case KEY_SHIFT:
                printf("Key Code %d , SHIFT is	ON\n", NewKey);
                break;
            case (KEY_SHIFT | 0x80):
                if (IsLongPress == 0)
                {
                    printf("Key Code %d ,	SHIFT IS OFF\n", NewKey);
                }
                break;
            case COMBINATION_KEY1:        // 如果是组合键1按下
                // 组合键按下的操作放在这里
                printf("Key Code %d , SHIFT-PLUS is ON\n", NewKey);
                if (IsLongPress == 1)        // 如果是组合键长按键
                {
                    while (IsLongPress == 1)        // 检查长按键是否依然有效
                    {
                        // 组合键长按键处理放在这里
                        printf("SHIFT-PLUS long	pressed\n");
                        delay_1ms(100);
                    }
                }
                break;
            case COMBINATION_KEY2:        // 如果是组合键2按下
                // 组合键2按下的处理放这里
                printf("Key Code %d , SHIFT-MINUS ON\n", NewKey);
                break;
            default:
                break;
            }
            if (NewKey & 0x80)        // 如果是按键释放 (所有键)
            {
                if (IsLongPress)        // 如果按键释放时间达到一个长按键时长
                {
                    if (++IdleCounter > 10)        // 无按键时间超过10个长按键时间(30秒)
                    {
                        // 超过30秒键盘无动作时的处理放这里
                        printf("There's	no keyboard operation for more than 30s.\n");
                    }
                }
            }
            NewKey = 0xFF;        // 清除按键值
        }
        // ********** 按键处理部分结束 ***********
    }
}

void io_init(void)        // 设置 UART 所用I/O口
{
    PORTB.OUTSET   = 0x04;        // 设置 TX(PB2) 为高电平
    PORTB.DIRSET   = 0x04;        // 设置 TX(PB2) 为输出
    PORTB.DIRCLR   = 0x08;        // 设置 RX(PB3) 为输入
    PORTB.PIN3CTRL = 0x08;        // 设置 RX(PB3) 上拉
}

void uart_init(void)        // 设置 UART 工作模式和波特率
{
    USART0.BAUD  = 0x56D;        // ATTINY416默认工作主频3.333MHz，设置串口在普通工作模式下波特率为9600
    USART0.CTRLC = 0x03;         // 设置为 异步模式、无奇偶校验、1个停止位、8位数据位
    USART0.CTRLB = 0xC0;         // 使能接收和发送
    USART0.CTRLA = 0x80;         // 使能 RXCIE(接收)中断
}

void timer_init(void)        // 设置 RTC 定时器
{
    RTC.CLKSEL     = 0x01;        // 选择 1KHz 内部时钟为信号源
    RTC.PITINTCTRL = 0x01;        // 使能周期中断
    while (RTC.PITSTATUS)
        ;                       // 确定 PITCTRLA 寄存器不忙
    RTC.PITCTRLA = 0x29;        // 使能周期中断功能，定为64时钟周期 (用作主程序中延时计时)
    while (RTC.STATUS & 0x01)
        ;                    // 确定 CTRLA 寄存器不忙
    RTC.CTRLA = 0x01;        // 设置分频率为 1, 使能 RTC
}

void delay_64ms(unsigned char T)
{
    DownCounter = T;
    while (0 != DownCounter)
        ;
}

// RTC 周期中断处理 (每64ms发生一次)
ISR(RTC_PIT_vect)
{
    RTC.PITINTFLAGS = 0x01;        // 清除中断标志
    if (0 != DownCounter)          // 如果 DownCounter 不为0，每64ms减一
    {
        --DownCounter;
    }
    if (++Counter_3s >= 47)        // 长按键定时，此处47*64ms约为3s
    {
        Counter_3s = 0;
        NewKey     = LastKeyNumber;        // 此时意味着最后一次按键变化的状态已经保持了设定的长按键时间(2s)没有变化
        // 重新向主程序发出键值
        IsLongPress = 1;        // 设置长按键标志
    }
}

// UART 接收中断处理 (按键处理)
ISR(USART0_RXC_vect)
{
    NewKey      = USART0.RXDATAL;        // 获取键值，同时清除中断标志
    Counter_3s  = 0;
    IsLongPress = 0;        // 按键状态发生改变，清除长按标志
    IdleCounter = 0;        // 有按键，休眠计时器清零
    while (RTC.STATUS & 0x02)
        ;               // 确定计数器 CNT 可写
    RTC.CNT = 0;        // 重新开始计时(长按键时间)

    // ***** 以下为组合键处理，如果不使用组合键，以下部分可以省略
    if (NewKey == CB1_KEY1)        // CB1_KEY1 按下
    {
        Cb1.key1 = 1;
    }
    if (NewKey == (CB1_KEY1 | 0x80))        // CB1_KEY1 释放
    {
        Cb1.key1 = 0;
    }

    if (NewKey == CB1_KEY2)        // CB1_KEY2 按下
    {
        Cb1.key2 = 1;
    }
    if (NewKey == (CB1_KEY2 | 0x80))        // CB1_KEY2 释放
    {
        Cb1.key2 = 0;
    }

    if (NewKey == CB2_KEY1)        // CB2_KEY1 按下
    {
        Cb2.key1 = 1;
    }
    if (NewKey == (CB2_KEY1 | 0x80))        // CB2_KEY1 释放
    {
        Cb2.key1 = 0;
    }

    if (NewKey == CB2_KEY2)        // CB2_KEY2 按下
    {
        Cb2.key2 = 1;
    }
    if (NewKey == (CB2_KEY2 | 0x80))        // CB2_KEY2 释放
    {
        Cb2.key2 = 0;
    }

    if (Cb1.key1 && Cb1.key2)        // 如果组合键1的两个键都为按下状态
    {
        NewKey = COMBINATION_KEY1;        // 键值替换为组合键的键值
    }
    if (Cb2.key1 && Cb2.key2)        // 如果组合键2的两个键都为按下状态
    {
        NewKey = COMBINATION_KEY2;        // 键值替换为组合键的键值
    }
    // ***** 组合键处理部分结束

    LastKeyNumber = NewKey;
}
